#
# poincare.py
#
"""
This program computes the Poincare section corresponding to a solution of 
Rossler equation read from data/solution.dat
Poincare section basis are defined by the unit vector in z direction and
e_2 = R_z*e_x where R_z is the rotation matrix about z axis.
"""
from scipy import interpolate

import numpy as np
import rossler
import solver

zerotol = 1e-12 #Zero tolerance for being on the Poincare section
	
def Rz(theta):
	R = np.array([[np.cos(theta), -np.sin(theta), 0],
				  [np.sin(theta),  np.cos(theta), 0],
				  [0,			   0,			  1]], float)
	return R
		
def compute(tx, theta=0, direction=1):
	"""
	Takes the solution and returns the corresponding Poincare section:
	"""
	xsol = tx[:,1:4]
	e_z = np.array([0,0,1], float) #Basis vector in z direction
	e_x = np.array([1,0,0], float) #Basis vector in x direction 
	e_2 = np.dot(Rz(theta), e_x)  #Basis vector on the Poincare section
	nhat = np.cross(e_z, e_2) #Normal vector to the Poincare section 

	#Poincare section condition:
	xhat = np.array([0,0,0],float)
	cond = np.dot(xsol-xhat, nhat)*direction

	#Function form:
	def U(x, xhat=np.array([0,0,0],float), nhat=nhat):
		cond = np.dot(x-xhat, nhat)*direction
		return cond

	ps = np.zeros([1,4])
	
	for i in range(len(cond) - 1):
		if cond[i] < 0 and cond[i+1] > 0: #If there is a zero-crossing
			
			tpsect = tx[i,0]
			xcandidate = xsol[i,:]
			deltat = (tx[i+1,0]-tx[i,0])/2
			psectcond = U(xcandidate)
			
			#Adaptive integration with smaller steps until the zero tolerance
			#is matched:
			while np.abs(psectcond) > zerotol:
						
				tint = [float(0), deltat]
				xint = solver.integrate(xcandidate, tint, abserror=1e-14, relerror=1e-12)
				
				xcandidate = xint[1,:]
				psectcond = U(xcandidate)
				tpsect = tpsect + deltat
				if psectcond > 0: #Go back and reduce the step size:
					
					tpsect = tpsect - deltat
					xcandidate = xint[0, :]
					psectcond = U(xcandidate)
					deltat = deltat/2
			
			#Once the zero tolerance is satisfied, store the point and the 
			#time:
			
			ps = np.append(ps, np.array([np.append(tpsect, xcandidate)]), axis=0)
	
	ps = ps[1:np.size(ps,0), :]

	#Matrix holding the basis vectors within the Poincare section and the normal:
	Vbasis = np.array([e_2, e_z, nhat])
	np.savetxt("data/Vbasis.dat", Vbasis)
	
	return ps
	
#If poincare.py is called as "main":
if __name__ == "__main__":

	#Read the solution:
	tx = np.loadtxt("data/solution.dat")
	theta=np.array([0], float)
	np.savetxt('data/theta.dat', theta)
	
	#Compute the Poincare section:
	ps = compute(tx, theta)
	
	Vbasis = np.loadtxt('data/Vbasis.dat')
	
	#Project the Poincare section onto this basis:
	psprojected = np.dot(Vbasis, ps[:, 1:4].transpose()).transpose()[:, 0:2]

	#Save the findings:	
	np.savetxt("data/psect.dat", ps)
	np.savetxt("data/psprojected.dat", psprojected)
	
	#Plot the projected Poincare section:
	
	from pylab import figure, plot, xlabel, ylabel, hold, savefig
	from matplotlib.font_manager import FontProperties
	import matplotlib as mpl
	import matplotlib.pyplot as plt
	
	mpl.rcParams['text.usetex']=True
	mpl.rcParams['text.latex.unicode']=True
	
	figure(1, figsize=(6, 6))
	
	xlabel('$e_2$', fontsize=18)
	ylabel('$e_z$', fontsize=18)
	
	plot(psprojected[:,0],psprojected[:,1], '.', ms=7)
	savefig("plot/rosslerpoincare.png", bbox_inches='tight', dpi=100)
	
	plt.tight_layout()
	plt.show()
