#
# returnmap.py
#
"""
This program computes the arclength return map corresponding to the Poincare
section computed in poincare.py and outputs periodic orbit candidates.
"""
import numpy as np
from scipy import interpolate
from scipy.optimize import newton, fsolve, root

import solver, poincare

theta = np.loadtxt('data/theta.dat')
direction = 1

#Load the Poincare section data:

ps = np.loadtxt("data/psect.dat")
Vbasis = np.loadtxt("data/Vbasis.dat")
psprojected = np.loadtxt("data/psprojected.dat")

#In order to find arc lengths along the Poincare section curve, start with
#Picking the point with the smallest e_2 component as the first point and
#sort the points from closest to the furthest to this point along the curve:

imin = np.argmin(psprojected[:,0])
pssorted = np.array([psprojected[imin,:]], float)
isorted = np.array([imin], int)

ssorted = np.array([0], float) #Array to hold arclengths

for j in range(np.size(psprojected,0)-1):
	
	diff = psprojected - pssorted[np.size(pssorted,0)-1,:]
	absdiff = np.linalg.norm(diff, axis=1)
	absdiff[isorted] = np.amax(absdiff) #Discard previously sorted points from the minimum distance search
	imin = np.argmin(absdiff)
	
	isorted = np.append(isorted, np.array([imin], int))
	pssorted = np.append(pssorted, np.array([psprojected[imin,:]], float), axis=0)
	sdiff = np.linalg.norm(pssorted[np.size(pssorted,0)-1,:] - pssorted[np.size(pssorted,0)-2,:])
	ssorted = np.append(ssorted, np.array([sdiff + ssorted[np.size(ssorted,0)-1]], float))

sps = np.zeros(np.size(ssorted,0))
sps[isorted] = ssorted #Arclengths in the order of projected Poincare section elements

#Return map data:
sn=sps[0:len(sps)-2]
issort = np.argsort(sn)
snsorted = sn[issort]
snplus1=sps[1:len(sps)-1]
snplus1sorted = snplus1[issort]

#Interpolation to the return map:

tck = interpolate.splrep(snsorted,snplus1sorted)

smax = np.amax(snsorted)
smin = np.amin(snsorted)
snint = np.arange(smin, smax, (smax-smin)/1000)
snplus1int = interpolate.splev(snint, tck)

#return map function:
def retmap(sn):
	snplus1 = interpolate.splev(sn, tck)
	return snplus1

#mth return map function:

def retmapm(m, sn):
	snplusm = retmap(sn)
	for i in range(m - 1):
		snplusm = retmap(snplusm)
	return	snplusm


#Plot the return map:

from pylab import figure, plot, xlabel, ylabel, hold, savefig
from matplotlib.font_manager import FontProperties
import matplotlib as mpl
import matplotlib.pyplot as plt

mpl.rcParams['text.usetex']=True
mpl.rcParams['text.latex.unicode']=True

figure(1, figsize=(6,6))

xlabel('$s_{n}$', fontsize=18)
ylabel('$s_{n+1}$', fontsize=18)

plot(sn,snplus1, '.', ms=7)

plt.hold(True)

plot(snint,snplus1int, 'k')
plot(snint, snint, 'g')


savefig("plot/rosslerretmap.png", bbox_inches='tight', dpi=100)

#Plot another return map:

#figure(2, figsize=(6,6))

#xlabel('$s_{n}$', fontsize=18)
#ylabel('$s_{n+6}$', fontsize=18)
 
#snplus6 = np.array([retmapm(6, sevl) for sevl in snint], float)

##plot(sn,snplus5, '.', ms=7)

#plt.hold(True)

#plot(snint,snplus6, 'k')
#plot(snint, snint, 'g')

#savefig("plot/rosslerretmap6.png", bbox_inches='tight', dpi=100)

plt.tight_layout()
plt.show()
