#
# invpolsolver.py
#
"""
Use odeint to solve differential equations defined by v in rossler.py
"""

from __future__ import unicode_literals
from scipy.integrate import odeint
import rossler
import numpy as np

#Callable function version:
def integrate(x0, t, p=[0.2, 0.2, 5.7], abserror=1.0e-8, relerror=1.0e-6):
	"""
	Takes the initial condition, parameters and the time interval
	returns the result as a series in time.
	ChaosBook Parameters [a=0.2, b=0.2, c=5.7] are assigned if parameters 
	are not specified otherwise
	"""
	xsol = odeint(rossler.v, x0, t, args=(p,), atol = abserror, rtol = relerror)
	
	return xsol

#If the solver is called as "main":
if __name__ == "__main__":
			
	stoptime = 1000
	numpoints = 100001
	
	# Create the time samples for the output of the ODE solver:
	t = [stoptime * float(i) / (numpoints - 1) for i in range(numpoints)]
	
	# Pack up the initial conditions (picked on the strange attractor):
	x0 = [-1.0027854, -8.6803028, 0.0260447]
	
	# Call the ODE solver
	xsol = integrate(x0,t)

	# Write the solution into the file data/solution.dat
	# Use numpy.savetxt (practical, faster):
	tx = np.array([t], float).transpose() #Add time to the first collumn of tx
	tx = np.append(tx, xsol, axis=1) #Add solution to the following collumns of tx
	np.savetxt("data/solution.dat", tx) #Write tx in data/solution.dat
	
	#Use file module (more general, enables custom formatting):
	#f = open("data/solution.dat", "w")
	
	#for i in range(len(t)):
		
		#f.write("%5.7f " % float(t[i])) # Write time
		#for j in range(np.size(xsol, 1)):
			#f.write("\t %5.7f" % float(xsol[i][j])) #Write coordinates delimited by \t characters
		#f.write("\n") #New line

	#f.close()

	# Plot the solution:
	#Import plotting modules:
	from pylab import figure, grid, hold, legend, savefig
	import matplotlib as mpl
	from mpl_toolkits.mplot3d import Axes3D
	import matplotlib.pyplot as plt 

	mpl.rcParams['text.usetex']=True
	mpl.rcParams['text.latex.unicode']=True

	fig = plt.figure() #Create a figure instance
	ax = fig.gca(projection = "3d") #Create an axis instance
	
	ax.plot(xsol[:,0], xsol[:,1], xsol[:,2], lw=0.5)
	ax.set_xlabel('$x$', fontsize=18)
	ax.set_ylabel('$y$', fontsize=18)
	ax.set_zlabel('$z$', fontsize=18)
	ax.view_init(30,-110)
	savefig("plot/rossler.png", bbox_inches='tight', dpi=100)

	plt.tight_layout()
	plt.show()
